import { Injectable, NotFoundException } from '@nestjs/common';
import { InjectRepository } from '@nestjs/typeorm';
import { Repository } from 'typeorm';
import { Reserva, TipoServico, StatusReserva } from './entities/reserva.entity';
import { CotacaoSeguro, TipoSeguro, StatusCotacao } from './entities/cotacao-seguro.entity';
import { CriarReservaDto, CotacaoSeguroDto } from './dto/servicos.dto';

@Injectable()
export class ServicosService {
  constructor(
    @InjectRepository(Reserva)
    private readonly reservaRepository: Repository<Reserva>,
    @InjectRepository(CotacaoSeguro)
    private readonly cotacaoRepository: Repository<CotacaoSeguro>,
  ) {}

  /**
   * Criar reserva (restaurante, hotel, guia turístico)
   */
  async criarReserva(userId: string, dto: CriarReservaDto) {
    const reserva = this.reservaRepository.create({
      clienteId: userId,
      tipoServico: dto.tipoServico,
      servicoNome: dto.servicoNome,
      dataReserva: dto.dataReserva,
      horario: dto.horario,
      numeroPessoas: dto.numeroPessoas,
      observacoes: dto.observacoes,
      detalhesAdicionais: dto.detalhesAdicionais,
      status: StatusReserva.PENDENTE,
    });

    await this.reservaRepository.save(reserva);

    // TODO: Enviar notificação ao prestador de serviço via Firebase
    console.log('📅 Nova reserva criada:', reserva.id);

    return {
      message: 'Reserva criada com sucesso',
      reserva: {
        id: reserva.id,
        servicoNome: reserva.servicoNome,
        dataReserva: reserva.dataReserva,
        status: reserva.status,
      },
    };
  }

  /**
   * Listar reservas do utilizador
   */
  async listarReservasUtilizador(userId: string) {
    const reservas = await this.reservaRepository.find({
      where: { clienteId: userId },
      order: { dataReserva: 'ASC' },
    });

    return {
      total: reservas.length,
      reservas,
    };
  }

  /**
   * Solicitar cotação de seguro (automóvel, residencial, vida)
   */
  async solicitarCotacaoSeguro(userId: string, dto: CotacaoSeguroDto) {
    const cotacao = this.cotacaoRepository.create({
      userId,
      tipo: dto.tipo,
      nomeCompleto: dto.nomeCompleto,
      telefone: dto.telefone,
      email: dto.email,
      veiculoMarca: dto.veiculoMarca,
      veiculoModelo: dto.veiculoModelo,
      veiculoAno: dto.veiculoAno,
      veiculoChaveta: dto.veiculoChaveta,
      status: StatusCotacao.PENDENTE,
    });

    // Calcular cotação simulada (em produção, integrar com API de seguradora)
    if (dto.tipo === TipoSeguro.AUTOMOVEL) {
      cotacao.valorMensal = this.calcularSeguroAutomovel(dto.veiculoAno || 2020);
      cotacao.valorAnual = cotacao.valorMensal * 12;
      cotacao.seguradoraParceira = 'ENSA Seguros';
    }

    await this.cotacaoRepository.save(cotacao);

    console.log('🛡️ Nova cotação de seguro solicitada:', cotacao.id);

    return {
      message: 'Cotação gerada com sucesso',
      cotacao: {
        id: cotacao.id,
        tipo: cotacao.tipo,
        valorMensal: cotacao.valorMensal,
        valorAnual: cotacao.valorAnual,
        seguradoraParceira: cotacao.seguradoraParceira,
      },
    };
  }

  /**
   * Obter detalhes de cotação
   */
  async obterCotacao(cotacaoId: string, userId: string) {
    const cotacao = await this.cotacaoRepository.findOne({
      where: { id: cotacaoId, userId },
    });

    if (!cotacao) {
      throw new NotFoundException('Cotação não encontrada');
    }

    return cotacao;
  }

  /**
   * Calcular valor simulado de seguro automóvel
   */
  private calcularSeguroAutomovel(ano: number): number {
    const anoAtual = new Date().getFullYear();
    const idade = anoAtual - ano;

    // Lógica simplificada (em produção, usar API real)
    let valorBase = 15000; // 15.000 Kz/mês base

    if (idade > 10) {
      valorBase += 5000; // Carros mais antigos custam mais
    }

    if (idade < 3) {
      valorBase += 8000; // Carros novos têm seguro mais caro
    }

    return valorBase;
  }
}
