// Teste de diagnóstico completo do backend TudoAqui
const http = require('http');

console.log('\n===================================');
console.log('🔍 DIAGNÓSTICO COMPLETO - TudoAqui');
console.log('===================================\n');

// Usar timestamp para garantir email e telefone únicos
const timestamp = Date.now();
const email = `teste${timestamp}@tudoaqui.ao`;
const telefone = `+244${timestamp.toString().substr(-9)}`;

console.log(`📧 Email: ${email}`);
console.log(`📱 Telefone: ${telefone}\n`);

// TESTE 1: Conexão básica ao backend
console.log('[1/4] Testando conexão ao backend...');
const healthReq = http.request({
  hostname: 'localhost',
  port: 3000,
  path: '/',
  method: 'GET'
}, (res) => {
  console.log(`✅ Backend acessível (Status: ${res.statusCode})\n`);
  
  // TESTE 2: CORS
  console.log('[2/4] Testando CORS...');
  const corsReq = http.request({
    hostname: 'localhost',
    port: 3000,
    path: '/api/v1/auth/register',
    method: 'OPTIONS',
    headers: {
      'Origin': 'http://localhost',
      'Access-Control-Request-Method': 'POST',
      'Access-Control-Request-Headers': 'Content-Type'
    }
  }, (res) => {
    console.log(`✅ CORS Status: ${res.statusCode}`);
    console.log(`✅ Allow-Origin: ${res.headers['access-control-allow-origin']}`);
    console.log(`✅ Allow-Methods: ${res.headers['access-control-allow-methods']}\n`);
    
    // TESTE 3: Registo de utilizador com tratamento de erros
    console.log('[3/4] Registando utilizador...');
    const postData = JSON.stringify({
      email: email,
      senha: 'Senha123!@',
      nome: 'Teste Sistema',
      telefone: telefone,
      tipo: 'vendedor'
    });
    
    console.log('📦 Payload:', JSON.stringify(JSON.parse(postData), null, 2));
    console.log('');
    
    const registerReq = http.request({
      hostname: 'localhost',
      port: 3000,
      path: '/api/v1/auth/register',
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Content-Length': Buffer.byteLength(postData),
        'Origin': 'http://localhost'
      }
    }, (res) => {
      let data = '';
      res.on('data', (chunk) => { data += chunk; });
      res.on('end', () => {
        console.log(`📡 Response Status: ${res.statusCode}`);
        console.log(`📡 Response Headers:`, JSON.stringify(res.headers, null, 2));
        console.log(`📡 Response Body:`, data);
        console.log('');
        
        if (res.statusCode === 201) {
          try {
            const response = JSON.parse(data);
            console.log('✅ REGISTO BEM-SUCEDIDO!');
            console.log(`   User ID: ${response.user.id}`);
            console.log(`   Email: ${response.user.email}`);
            console.log(`   Nome: ${response.user.nome}`);
            console.log(`   Tipo: ${response.user.tipo}`);
            console.log(`   Token: ${response.token.substring(0, 30)}...`);
            console.log('');
            
            // TESTE 4: Verificar status KYC
            console.log('[4/4] Verificando status KYC...');
            const statusReq = http.request({
              hostname: 'localhost',
              port: 3000,
              path: `/api/v1/kyc/status/${response.user.id}`,
              method: 'GET',
              headers: {
                'Authorization': `Bearer ${response.token}`
              }
            }, (res) => {
              let kycData = '';
              res.on('data', (chunk) => { kycData += chunk; });
              res.on('end', () => {
                if (res.statusCode === 200) {
                  const kyc = JSON.parse(kycData);
                  console.log(`✅ KYC Status Code: ${res.statusCode}`);
                  console.log(`✅ kycStatus: ${kyc.kycStatus}`);
                  console.log(`✅ seloVerificado: ${kyc.seloVerificado}`);
                  console.log(`✅ User Tipo: ${kyc.tipo}`);
                  console.log('');
                  console.log('===================================');
                  console.log('✅ TODOS OS TESTES PASSARAM!');
                  console.log('===================================');
                  console.log('');
                  console.log('✨ Sistema funcional e pronto para KYC!');
                } else {
                  console.error(`❌ KYC Error: ${res.statusCode}`);
                  console.error(`   Response: ${kycData}`);
                }
              });
            });
            
            statusReq.on('error', (e) => {
              console.error(`❌ Erro no request KYC: ${e.message}`);
            });
            statusReq.end();
            
          } catch (parseError) {
            console.error(`❌ Erro ao fazer parse da resposta: ${parseError.message}`);
            console.error(`   Raw data: ${data}`);
          }
        } else if (res.statusCode === 400) {
          console.error(`❌ ERRO DE VALIDAÇÃO (400)`);
          console.error(`   ${data}`);
          try {
            const error = JSON.parse(data);
            if (error.message && Array.isArray(error.message)) {
              console.error(`   Campos inválidos:`);
              error.message.forEach(msg => console.error(`   - ${msg}`));
            }
          } catch (e) {}
        } else if (res.statusCode === 500) {
          console.error(`❌ ERRO INTERNO DO SERVIDOR (500)`);
          console.error(`   ${data}`);
          console.error('');
          console.error('💡 Possíveis causas:');
          console.error('   - Telefone duplicado na base de dados');
          console.error('   - Constraint violation');
          console.error('   - PostgreSQL connection issue');
          console.error('   - Erro no bcrypt ou JWT');
        } else {
          console.error(`❌ Erro inesperado: ${res.statusCode}`);
          console.error(`   ${data}`);
        }
      });
    });
    
    registerReq.on('error', (e) => {
      console.error(`❌ Erro na conexão de registo: ${e.message}`);
      console.error(`   Código: ${e.code}`);
    });
    registerReq.write(postData);
    registerReq.end();
  });
  
  corsReq.on('error', (e) => {
    console.error(`❌ Erro no teste CORS: ${e.message}`);
  });
  corsReq.end();
});

healthReq.on('error', (e) => {
  console.error(`❌ Backend inacessível: ${e.message}`);
  console.error('⚠️ Certifique-se que o backend está a funcionar em http://localhost:3000');
  process.exit(1);
});
healthReq.end();
