#!/bin/bash
set -e

# Logging
exec > >(tee -a /var/log/tudoaqui-deploy.log)
exec 2>&1

echo "🚀 Iniciando deployment do TudoAqui em $(date)"

# ============================================
# Update do sistema
# ============================================
apt-get update
apt-get upgrade -y

# ============================================
# Instalar dependências do sistema
# ============================================
apt-get install -y \
    curl \
    wget \
    git \
    build-essential \
    software-properties-common \
    apt-transport-https \
    ca-certificates \
    gnupg \
    lsb-release

# ============================================
# Instalar Node.js 18
# ============================================
curl -fsSL https://deb.nodesource.com/setup_18.x | bash -
apt-get install -y nodejs

echo "✅ Node.js version: $(node -v)"
echo "✅ npm version: $(npm -v)"

# ============================================
# Instalar PostgreSQL Client (para testes)
# ============================================
apt-get install -y postgresql-client

# ============================================
# Instalar Redis CLI (para testes)
# ============================================
apt-get install -y redis-tools

# ============================================
# Instalar Nginx (reverse proxy)
# ============================================
apt-get install -y nginx

# ============================================
# Instalar PM2 (process manager)
# ============================================
npm install -g pm2
pm2 completion install

# ============================================
# Instalar Certbot (SSL/HTTPS)
# ============================================
apt-get install -y certbot python3-certbot-nginx

# ============================================
# Criar usuário tudoaqui
# ============================================
if ! id "tudoaqui" &>/dev/null; then
    useradd -m -s /bin/bash tudoaqui
fi

# ============================================
# Criar diretórios
# ============================================
mkdir -p /var/www/tudoaqui
mkdir -p /var/log/tudoaqui
mkdir -p /home/tudoaqui/.pm2

# ============================================
# Clonar repositório (substituir URL se necessário)
# ============================================
cd /var/www/tudoaqui

if [ ! -d ".git" ]; then
    git clone https://github.com/tudoaqui/backend.git . || {
        echo "⚠️ Repositório não encontrado, criando estrutura básica..."
        mkdir -p src dist
        npm init -y
    }
fi

# ============================================
# Instalar dependências Node.js
# ============================================
npm ci --production || npm install --production

# ============================================
# Build da aplicação
# ============================================
if [ -f "package.json" ]; then
    npm run build || echo "⚠️ Build falhou, continuando..."
fi

# ============================================
# Criar arquivo .env
# ============================================
cat > /var/www/tudoaqui/.env << 'ENVEOF'
# Ambiente
NODE_ENV=production
PORT=3000
LOG_LEVEL=info

# Database
DB_HOST=${db_host}
DB_PORT=5432
DB_USERNAME=${db_user}
DB_PASSWORD=${db_password}
DB_DATABASE=${db_name}
DB_SYNCHRONIZE=false

# Redis
REDIS_HOST=${redis_host}
REDIS_PORT=6379
REDIS_DB=0
REDIS_PASSWORD=

# JWT
JWT_SECRET=${jwt_secret}
JWT_EXPIRATION=24h
JWT_REFRESH_SECRET=${jwt_secret}-refresh
JWT_REFRESH_EXPIRATION=7d

# AWS
AWS_REGION=${aws_region}
AWS_S3_BUCKET=${s3_bucket}
AWS_ACCESS_KEY_ID=
AWS_SECRET_ACCESS_KEY=

# Integrations
SENDGRID_API_KEY=
FIREBASE_API_KEY=
AFRICA_TALKING_KEY=
REKOGNITION_ENABLED=true

# URLs
API_URL=https://api.tudoaqui.ao
FRONTEND_URL=https://tudoaqui.ao
WEBHOOK_URL=https://api.tudoaqui.ao/webhooks

# Segurança
CORS_ORIGIN=https://tudoaqui.ao
RATE_LIMIT_WINDOW=15
RATE_LIMIT_MAX_REQUESTS=100
ENVEOF

echo "✅ Arquivo .env criado"

# ============================================
# Configurar permissões
# ============================================
chown -R tudoaqui:tudoaqui /var/www/tudoaqui
chown -R tudoaqui:tudoaqui /var/log/tudoaqui
chmod -R 750 /var/www/tudoaqui

# ============================================
# Configurar Nginx como reverse proxy
# ============================================
cat > /etc/nginx/sites-available/tudoaqui << 'NGINXEOF'
upstream tudoaqui_app {
    server 127.0.0.1:3000;
    keepalive 32;
}

server {
    listen 80;
    server_name _;

    location /.well-known/acme-challenge {
        root /var/www/certbot;
    }

    location / {
        return 301 https://$host$request_uri;
    }
}

server {
    listen 443 ssl http2;
    server_name _;

    # SSL (substituir por certificado real)
    # ssl_certificate /etc/letsencrypt/live/api.tudoaqui.ao/fullchain.pem;
    # ssl_certificate_key /etc/letsencrypt/live/api.tudoaqui.ao/privkey.pem;

    # Security headers
    add_header Strict-Transport-Security "max-age=31536000; includeSubDomains" always;
    add_header X-Frame-Options "SAMEORIGIN" always;
    add_header X-Content-Type-Options "nosniff" always;
    add_header X-XSS-Protection "1; mode=block" always;
    add_header Referrer-Policy "strict-origin-when-cross-origin" always;

    # Gzip compression
    gzip on;
    gzip_vary on;
    gzip_min_length 1000;
    gzip_types text/plain text/css text/xml text/javascript application/x-javascript application/xml+rss;

    # Rate limiting
    limit_req_zone $binary_remote_addr zone=api_limit:10m rate=10r/s;
    limit_req zone=api_limit burst=20 nodelay;

    # Proxy settings
    location / {
        proxy_pass http://tudoaqui_app;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
        
        # Timeouts
        proxy_connect_timeout 60s;
        proxy_send_timeout 60s;
        proxy_read_timeout 60s;
    }

    # Health check
    location /health {
        access_log off;
        proxy_pass http://tudoaqui_app;
    }

    # Swagger docs
    location /api-docs {
        proxy_pass http://tudoaqui_app;
    }
}
NGINXEOF

# Ativar site
rm -f /etc/nginx/sites-enabled/default
ln -sf /etc/nginx/sites-available/tudoaqui /etc/nginx/sites-enabled/tudoaqui

# Validar configuração
nginx -t

# Iniciar Nginx
systemctl start nginx
systemctl enable nginx

echo "✅ Nginx configurado"

# ============================================
# Configurar PM2 para auto-start
# ============================================
cat > /etc/systemd/system/pm2-tudoaqui.service << 'PM2EOF'
[Unit]
Description=PM2 process manager for TudoAqui
After=network.target

[Service]
Type=forking
User=tudoaqui
WorkingDirectory=/var/www/tudoaqui
ExecStart=/usr/bin/pm2 start ecosystem.config.js --env production
ExecReload=/usr/bin/pm2 gracefulReload all
ExecStop=/usr/bin/pm2 stop all
Restart=on-failure
RestartSec=10

[Install]
WantedBy=multi-user.target
PM2EOF

# ============================================
# Criar ecosystem.config.js para PM2
# ============================================
cat > /var/www/tudoaqui/ecosystem.config.js << 'PMEOF'
module.exports = {
  apps: [
    {
      name: 'tudoaqui-api',
      script: './dist/main.js',
      instances: 'max',
      exec_mode: 'cluster',
      env: {
        NODE_ENV: 'production'
      },
      error_file: '/var/log/tudoaqui/error.log',
      out_file: '/var/log/tudoaqui/out.log',
      log_date_format: 'YYYY-MM-DD HH:mm:ss Z',
      merge_logs: true,
      autorestart: true,
      watch: false,
      max_memory_restart: '1G'
    }
  ]
};
PMEOF

chown tudoaqui:tudoaqui /var/www/tudoaqui/ecosystem.config.js

# ============================================
# Iniciar aplicação com PM2
# ============================================
cd /var/www/tudoaqui
sudo -u tudoaqui /usr/bin/pm2 start ecosystem.config.js --env production || echo "⚠️ Erro ao iniciar com PM2"
sudo -u tudoaqui /usr/bin/pm2 startup systemd -u tudoaqui --hp /home/tudoaqui

# Enable PM2 service
systemctl enable pm2-tudoaqui

echo "✅ PM2 configurado"

# ============================================
# Health check
# ============================================
echo "🏥 Realizando health checks..."
sleep 10

if curl -f http://localhost:3000/health > /dev/null 2>&1; then
    echo "✅ Aplicação respondendo em http://localhost:3000"
else
    echo "⚠️ Aplicação ainda não respondendo, esperar mais 30 segundos..."
    sleep 30
fi

# ============================================
# Setup SSL (manual - substituir domínio)
# ============================================
cat > /usr/local/bin/setup-ssl.sh << 'SSLEOF'
#!/bin/bash
DOMAIN=$1
if [ -z "$DOMAIN" ]; then
    echo "Uso: $0 seu-dominio.ao"
    exit 1
fi

certbot certonly --standalone -d $DOMAIN -d api.$DOMAIN
certbot install --nginx -d $DOMAIN -d api.$DOMAIN
systemctl reload nginx
SSLEOF

chmod +x /usr/local/bin/setup-ssl.sh

# ============================================
# Logs
# ============================================
echo ""
echo "📊 Deployment Summary:"
echo "===================="
echo "✅ Sistema operacional: $(lsb_release -ds)"
echo "✅ Node.js: $(node -v)"
echo "✅ npm: $(npm -v)"
echo "✅ Nginx: $(nginx -v 2>&1)"
echo "✅ PM2: $(pm2 -v)"
echo ""
echo "📝 Configuração:"
echo "  - Aplicação: /var/www/tudoaqui"
echo "  - Logs: /var/log/tudoaqui"
echo "  - Nginx: /etc/nginx/sites-available/tudoaqui"
echo "  - Env: /var/www/tudoaqui/.env"
echo ""
echo "🚀 Próximos passos:"
echo "  1. Configurar domínio DNS"
echo "  2. Setup SSL: setup-ssl.sh api.tudoaqui.ao"
echo "  3. Verificar status: pm2 status"
echo "  4. Ver logs: pm2 logs"
echo ""
echo "🎉 Deployment concluído em $(date)"
