import 'package:flutter/material.dart';
import '../../utils/constants.dart';

class RegistroScreen extends StatefulWidget {
  @override
  _RegistroScreenState createState() => _RegistroScreenState();
}

class _RegistroScreenState extends State<RegistroScreen> {
  final _nomeController = TextEditingController();
  final _emailController = TextEditingController();
  final _telefoneController = TextEditingController();
  final _senhaController = TextEditingController();
  final _confirmarSenhaController = TextEditingController();
  
  bool _obscureText = true;
  bool _confirmarObscureText = true;
  bool _isEmpresa = false;
  bool _isLoading = false;
  bool _concordaTermos = false;

  @override
  void dispose() {
    _nomeController.dispose();
    _emailController.dispose();
    _telefoneController.dispose();
    _senhaController.dispose();
    _confirmarSenhaController.dispose();
    super.dispose();
  }

  void _fazerRegistro() async {
    // Validações
    if (_nomeController.text.isEmpty ||
        _emailController.text.isEmpty ||
        _telefoneController.text.isEmpty ||
        _senhaController.text.isEmpty) {
      _mostrarErro('Por favor, preencha todos os campos');
      return;
    }

    if (!_emailController.text.contains('@')) {
      _mostrarErro('Email inválido');
      return;
    }

    if (_senhaController.text != _confirmarSenhaController.text) {
      _mostrarErro('As senhas não conferem');
      return;
    }

    if (_senhaController.text.length < 6) {
      _mostrarErro('A senha deve ter pelo menos 6 caracteres');
      return;
    }

    if (!_concordaTermos) {
      _mostrarErro('Deve concordar com os termos de serviço');
      return;
    }

    setState(() {
      _isLoading = true;
    });

    // Simular chamada ao backend (POST /auth/register)
    await Future.delayed(Duration(seconds: 2));

    setState(() {
      _isLoading = false;
    });

    // Navegar para KYC (verificação de documentos)
    Navigator.of(context).pushReplacementNamed('/verificacao');
  }

  void _mostrarErro(String mensagem) {
    ScaffoldMessenger.of(context).showSnackBar(
      SnackBar(
        content: Text(mensagem),
        backgroundColor: AppColors.danger,
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: Text('Criar Conta'),
        backgroundColor: AppColors.primary,
        elevation: 0,
      ),
      body: SafeArea(
        child: SingleChildScrollView(
          child: Padding(
            padding: EdgeInsets.all(AppDimensions.paddingScreen),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(
                  'Bem-vindo ao TudoAqui!',
                  style: AppTextStyles.heading2,
                ),
                SizedBox(height: 10),
                Text(
                  'Crie sua conta para começar a vender ou comprar.',
                  style: AppTextStyles.body,
                ),
                SizedBox(height: 30),

                // Nome
                _buildTextInputField(
                  label: 'Nome Completo',
                  controller: _nomeController,
                  icon: Icons.person,
                  hintText: 'João Silva',
                ),

                SizedBox(height: 20),

                // Email
                _buildTextInputField(
                  label: 'Email',
                  controller: _emailController,
                  icon: Icons.email,
                  hintText: 'seu.email@exemplo.com',
                  keyboardType: TextInputType.emailAddress,
                ),

                SizedBox(height: 20),

                // Telefone
                _buildTextInputField(
                  label: 'Telefone',
                  controller: _telefoneController,
                  icon: Icons.phone,
                  hintText: '+244 923 123 456',
                  keyboardType: TextInputType.phone,
                ),

                SizedBox(height: 20),

                // Toggle Empresa
                Container(
                  padding: EdgeInsets.all(15),
                  decoration: BoxDecoration(
                    color: Colors.blue[50],
                    borderRadius: BorderRadius.circular(10),
                  ),
                  child: Row(
                    children: [
                      Expanded(
                        child: Text(
                          'Registar como Empresa?',
                          style: TextStyle(
                            fontWeight: FontWeight.w600,
                            fontSize: 16,
                          ),
                        ),
                      ),
                      Switch(
                        value: _isEmpresa,
                        onChanged: (value) {
                          setState(() {
                            _isEmpresa = value;
                          });
                        },
                        activeColor: AppColors.primary,
                      ),
                    ],
                  ),
                ),

                SizedBox(height: 20),

                // Senha
                _buildPasswordField(
                  label: 'Senha',
                  controller: _senhaController,
                  obscureText: _obscureText,
                  onToggle: () {
                    setState(() {
                      _obscureText = !_obscureText;
                    });
                  },
                  hintText: '••••••••',
                ),

                SizedBox(height: 20),

                // Confirmar Senha
                _buildPasswordField(
                  label: 'Confirmar Senha',
                  controller: _confirmarSenhaController,
                  obscureText: _confirmarObscureText,
                  onToggle: () {
                    setState(() {
                      _confirmarObscureText = !_confirmarObscureText;
                    });
                  },
                  hintText: '••••••••',
                ),

                SizedBox(height: 20),

                // Termos de Serviço
                Row(
                  children: [
                    Checkbox(
                      value: _concordaTermos,
                      onChanged: (value) {
                        setState(() {
                          _concordaTermos = value ?? false;
                        });
                      },
                      activeColor: AppColors.primary,
                    ),
                    Expanded(
                      child: GestureDetector(
                        onTap: () {
                          // TODO: Abrir termos de serviço
                        },
                        child: RichText(
                          text: TextSpan(
                            text: 'Concordo com os ',
                            style: AppTextStyles.caption,
                            children: [
                              TextSpan(
                                text: 'Termos de Serviço',
                                style: TextStyle(
                                  color: AppColors.primary,
                                  fontWeight: FontWeight.bold,
                                  decoration: TextDecoration.underline,
                                ),
                              ),
                            ],
                          ),
                        ),
                      ),
                    ),
                  ],
                ),

                SizedBox(height: 30),

                // Botão Registar
                SizedBox(
                  width: double.infinity,
                  height: AppDimensions.buttonHeight,
                  child: ElevatedButton(
                    style: ElevatedButton.styleFrom(
                      backgroundColor: AppColors.primary,
                      shape: RoundedRectangleBorder(
                        borderRadius: BorderRadius.circular(AppDimensions.borderRadiusMedium),
                      ),
                    ),
                    onPressed: _isLoading ? null : _fazerRegistro,
                    child: _isLoading
                        ? SizedBox(
                            height: 20,
                            width: 20,
                            child: CircularProgressIndicator(
                              valueColor: AlwaysStoppedAnimation<Color>(Colors.white),
                              strokeWidth: 2,
                            ),
                          )
                        : Text(
                            'CRIAR CONTA',
                            style: TextStyle(
                              color: Colors.white,
                              fontWeight: FontWeight.bold,
                              fontSize: 16,
                            ),
                          ),
                  ),
                ),

                SizedBox(height: 20),

                // Voltar para Login
                Center(
                  child: GestureDetector(
                    onTap: () {
                      Navigator.of(context).pop();
                    },
                    child: RichText(
                      text: TextSpan(
                        text: 'Já tem conta? ',
                        style: AppTextStyles.body,
                        children: [
                          TextSpan(
                            text: 'Entrar',
                            style: TextStyle(
                              color: AppColors.primary,
                              fontWeight: FontWeight.bold,
                            ),
                          ),
                        ],
                      ),
                    ),
                  ),
                ),

                SizedBox(height: 40),
              ],
            ),
          ),
        ),
      ),
    );
  }

  Widget _buildTextInputField({
    required String label,
    required TextEditingController controller,
    required IconData icon,
    required String hintText,
    TextInputType keyboardType = TextInputType.text,
  }) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          label,
          style: TextStyle(
            fontSize: 14,
            fontWeight: FontWeight.w600,
            color: AppColors.textDark,
          ),
        ),
        SizedBox(height: 10),
        TextField(
          controller: controller,
          keyboardType: keyboardType,
          decoration: InputDecoration(
            hintText: hintText,
            prefixIcon: Icon(icon, color: AppColors.primary),
            border: OutlineInputBorder(
              borderRadius: BorderRadius.circular(10),
              borderSide: BorderSide(color: Colors.grey[300]!),
            ),
            focusedBorder: OutlineInputBorder(
              borderRadius: BorderRadius.circular(10),
              borderSide: BorderSide(color: AppColors.primary, width: 2),
            ),
          ),
        ),
      ],
    );
  }

  Widget _buildPasswordField({
    required String label,
    required TextEditingController controller,
    required bool obscureText,
    required VoidCallback onToggle,
    required String hintText,
  }) {
    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          label,
          style: TextStyle(
            fontSize: 14,
            fontWeight: FontWeight.w600,
            color: AppColors.textDark,
          ),
        ),
        SizedBox(height: 10),
        TextField(
          controller: controller,
          obscureText: obscureText,
          decoration: InputDecoration(
            hintText: hintText,
            prefixIcon: Icon(Icons.lock, color: AppColors.primary),
            suffixIcon: IconButton(
              icon: Icon(
                obscureText ? Icons.visibility_off : Icons.visibility,
                color: AppColors.primary,
              ),
              onPressed: onToggle,
            ),
            border: OutlineInputBorder(
              borderRadius: BorderRadius.circular(10),
              borderSide: BorderSide(color: Colors.grey[300]!),
            ),
            focusedBorder: OutlineInputBorder(
              borderRadius: BorderRadius.circular(10),
              borderSide: BorderSide(color: AppColors.primary, width: 2),
            ),
          ),
        ),
      ],
    );
  }
}
