import 'package:flutter/material.dart';
import '../../utils/constants.dart';

class ListaRestaurantes extends StatefulWidget {
  @override
  _ListaRestaurantesState createState() => _ListaRestaurantesState();
}

class _ListaRestaurantesState extends State<ListaRestaurantes> {
  // Dados simulados - virão da API
  final List<Map<String, dynamic>> restaurantes = [
    {
      'nome': 'Restaurante Chez Wou',
      'foto': 'https://via.placeholder.com/400x200',
      'rating': 4.5,
      'distancia': '2.3 km',
      'tempoEntrega': '25-35 min',
      'entregaDisponivel': true,
      'parceiros': ['Tupuca'],
    },
    {
      'nome': 'Churrascaria Bom Apetite',
      'foto': 'https://via.placeholder.com/400x200',
      'rating': 4.8,
      'distancia': '1.8 km',
      'tempoEntrega': '20-30 min',
      'entregaDisponivel': true,
      'parceiros': ['Tupuca', 'Mano'],
    },
    {
      'nome': 'Pizza da Casa',
      'foto': 'https://via.placeholder.com/400x200',
      'rating': 4.2,
      'distancia': '3.5 km',
      'tempoEntrega': '30-40 min',
      'entregaDisponivel': true,
      'parceiros': ['Tupuca'],
    },
  ];

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: Text("Restaurantes"),
        backgroundColor: AppColors.primary,
      ),
      body: Column(
        children: [
          _buildFiltros(),
          Expanded(
            child: ListView.builder(
              padding: EdgeInsets.all(AppDimensions.paddingScreen),
              itemCount: restaurantes.length,
              itemBuilder: (context, index) {
                return _buildCardRestaurante(restaurantes[index]);
              },
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildFiltros() {
    return Container(
      padding: EdgeInsets.all(15),
      color: Colors.white,
      child: Row(
        children: [
          Expanded(
            child: TextField(
              decoration: InputDecoration(
                hintText: "Procurar restaurantes...",
                prefixIcon: Icon(Icons.search),
                border: OutlineInputBorder(
                  borderRadius: BorderRadius.circular(10),
                  borderSide: BorderSide.none,
                ),
                filled: true,
                fillColor: Colors.grey[200],
                contentPadding: EdgeInsets.symmetric(vertical: 0),
              ),
            ),
          ),
          SizedBox(width: 10),
          IconButton(
            icon: Icon(Icons.filter_list, color: AppColors.primary),
            onPressed: () {
              // Modal de filtros
            },
          ),
        ],
      ),
    );
  }

  Widget _buildCardRestaurante(Map<String, dynamic> restaurante) {
    return GestureDetector(
      onTap: () {
        // Navegar para o menu do restaurante
        _abrirMenuRestaurante(restaurante);
      },
      child: Container(
        margin: EdgeInsets.only(bottom: 20),
        decoration: BoxDecoration(
          color: Colors.white,
          borderRadius: BorderRadius.circular(AppDimensions.borderRadiusLarge),
          boxShadow: [
            BoxShadow(
              color: Colors.black12,
              blurRadius: 8,
              offset: Offset(0, 2),
            ),
          ],
        ),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            // Foto do Restaurante
            ClipRRect(
              borderRadius: BorderRadius.vertical(
                top: Radius.circular(AppDimensions.borderRadiusLarge),
              ),
              child: Image.network(
                restaurante['foto'],
                height: 150,
                width: double.infinity,
                fit: BoxFit.cover,
                errorBuilder: (context, error, stackTrace) {
                  return Container(
                    height: 150,
                    color: Colors.grey[300],
                    child: Icon(Icons.restaurant, size: 50, color: Colors.grey),
                  );
                },
              ),
            ),
            
            Padding(
              padding: EdgeInsets.all(15),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  // Nome e Rating
                  Row(
                    mainAxisAlignment: MainAxisAlignment.spaceBetween,
                    children: [
                      Expanded(
                        child: Text(
                          restaurante['nome'],
                          style: TextStyle(
                            fontSize: 18,
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ),
                      Row(
                        children: [
                          Icon(Icons.star, color: Colors.amber, size: 18),
                          SizedBox(width: 4),
                          Text(
                            restaurante['rating'].toString(),
                            style: TextStyle(
                              fontSize: 14,
                              fontWeight: FontWeight.w600,
                            ),
                          ),
                        ],
                      ),
                    ],
                  ),
                  
                  SizedBox(height: 10),
                  
                  // Distância e Tempo de Entrega
                  Row(
                    children: [
                      Icon(Icons.location_on, size: 16, color: Colors.grey),
                      SizedBox(width: 4),
                      Text(
                        restaurante['distancia'],
                        style: AppTextStyles.caption,
                      ),
                      SizedBox(width: 15),
                      Icon(Icons.access_time, size: 16, color: Colors.grey),
                      SizedBox(width: 4),
                      Text(
                        restaurante['tempoEntrega'],
                        style: AppTextStyles.caption,
                      ),
                    ],
                  ),
                  
                  SizedBox(height: 10),
                  
                  // Badge de Parceiros de Entrega
                  if (restaurante['entregaDisponivel'])
                    Wrap(
                      spacing: 8,
                      children: [
                        for (var parceiro in restaurante['parceiros'])
                          Container(
                            padding: EdgeInsets.symmetric(horizontal: 10, vertical: 5),
                            decoration: BoxDecoration(
                              color: AppColors.primary.withOpacity(0.1),
                              borderRadius: BorderRadius.circular(20),
                            ),
                            child: Row(
                              mainAxisSize: MainAxisSize.min,
                              children: [
                                Icon(
                                  Icons.local_shipping,
                                  size: 14,
                                  color: AppColors.primary,
                                ),
                                SizedBox(width: 4),
                                Text(
                                  "Entrega por $parceiro",
                                  style: TextStyle(
                                    fontSize: 11,
                                    color: AppColors.primary,
                                    fontWeight: FontWeight.w600,
                                  ),
                                ),
                              ],
                            ),
                          ),
                      ],
                    ),
                ],
              ),
            ),
          ],
        ),
      ),
    );
  }

  void _abrirMenuRestaurante(Map<String, dynamic> restaurante) {
    Navigator.push(
      context,
      MaterialPageRoute(
        builder: (context) => MenuRestaurante(restaurante: restaurante),
      ),
    );
  }
}

// Ecrã de Menu do Restaurante
class MenuRestaurante extends StatefulWidget {
  final Map<String, dynamic> restaurante;

  MenuRestaurante({required this.restaurante});

  @override
  _MenuRestauranteState createState() => _MenuRestauranteState();
}

class _MenuRestauranteState extends State<MenuRestaurante> {
  List<Map<String, dynamic>> carrinho = [];

  // Menu simulado
  final List<Map<String, dynamic>> pratos = [
    {
      'nome': 'Muamba de Galinha',
      'descricao': 'Prato tradicional angolano com arroz e funge',
      'preco': 3500,
      'foto': 'https://via.placeholder.com/150',
    },
    {
      'nome': 'Calulu de Peixe',
      'descricao': 'Peixe fresco com legumes e óleo de palma',
      'preco': 4000,
      'foto': 'https://via.placeholder.com/150',
    },
    {
      'nome': 'Churrasco Misto',
      'descricao': 'Picanha, frango e linguiça com batatas',
      'preco': 5500,
      'foto': 'https://via.placeholder.com/150',
    },
  ];

  @override
  Widget build(BuildContext context) {
    double totalCarrinho = carrinho.fold(0, (sum, item) => sum + item['preco']);

    return Scaffold(
      appBar: AppBar(
        title: Text(widget.restaurante['nome']),
        backgroundColor: AppColors.primary,
      ),
      body: Column(
        children: [
          // Info do Restaurante
          Container(
            padding: EdgeInsets.all(20),
            color: Colors.white,
            child: Row(
              children: [
                Expanded(
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    children: [
                      Text(
                        "Entrega garantida por:",
                        style: AppTextStyles.caption,
                      ),
                      SizedBox(height: 5),
                      Row(
                        children: [
                          Container(
                            padding: EdgeInsets.symmetric(horizontal: 12, vertical: 6),
                            decoration: BoxDecoration(
                              color: AppColors.primary,
                              borderRadius: BorderRadius.circular(20),
                            ),
                            child: Text(
                              "🚚 Tupuca",
                              style: TextStyle(
                                color: Colors.white,
                                fontWeight: FontWeight.bold,
                              ),
                            ),
                          ),
                        ],
                      ),
                    ],
                  ),
                ),
                Column(
                  crossAxisAlignment: CrossAxisAlignment.end,
                  children: [
                    Text(
                      widget.restaurante['tempoEntrega'],
                      style: TextStyle(fontWeight: FontWeight.bold),
                    ),
                    Text(
                      "Tempo estimado",
                      style: AppTextStyles.caption,
                    ),
                  ],
                ),
              ],
            ),
          ),
          
          // Lista de Pratos
          Expanded(
            child: ListView.builder(
              padding: EdgeInsets.all(20),
              itemCount: pratos.length,
              itemBuilder: (context, index) {
                return _buildCardPrato(pratos[index]);
              },
            ),
          ),
          
          // Carrinho (se tiver itens)
          if (carrinho.isNotEmpty)
            Container(
              padding: EdgeInsets.all(20),
              decoration: BoxDecoration(
                color: Colors.white,
                boxShadow: [
                  BoxShadow(
                    color: Colors.black12,
                    blurRadius: 8,
                    offset: Offset(0, -2),
                  ),
                ],
              ),
              child: SafeArea(
                child: ElevatedButton(
                  style: ElevatedButton.styleFrom(
                    backgroundColor: AppColors.primary,
                    minimumSize: Size(double.infinity, 55),
                    shape: RoundedRectangleBorder(
                      borderRadius: BorderRadius.circular(10),
                    ),
                  ),
                  onPressed: () {
                    _finalizarPedido();
                  },
                  child: Row(
                    mainAxisAlignment: MainAxisAlignment.spaceBetween,
                    children: [
                      Text(
                        "Ver Carrinho (${carrinho.length})",
                        style: TextStyle(color: Colors.white, fontSize: 16),
                      ),
                      Text(
                        "${totalCarrinho.toStringAsFixed(0)} Kz",
                        style: TextStyle(
                          color: Colors.white,
                          fontSize: 18,
                          fontWeight: FontWeight.bold,
                        ),
                      ),
                    ],
                  ),
                ),
              ),
            ),
        ],
      ),
    );
  }

  Widget _buildCardPrato(Map<String, dynamic> prato) {
    return Container(
      margin: EdgeInsets.only(bottom: 15),
      decoration: BoxDecoration(
        color: Colors.white,
        borderRadius: BorderRadius.circular(10),
        boxShadow: [
          BoxShadow(
            color: Colors.black12,
            blurRadius: 4,
            offset: Offset(0, 2),
          ),
        ],
      ),
      child: Row(
        children: [
          ClipRRect(
            borderRadius: BorderRadius.horizontal(left: Radius.circular(10)),
            child: Image.network(
              prato['foto'],
              width: 100,
              height: 100,
              fit: BoxFit.cover,
              errorBuilder: (context, error, stackTrace) {
                return Container(
                  width: 100,
                  height: 100,
                  color: Colors.grey[300],
                  child: Icon(Icons.restaurant_menu),
                );
              },
            ),
          ),
          Expanded(
            child: Padding(
              padding: EdgeInsets.all(12),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(
                    prato['nome'],
                    style: TextStyle(
                      fontSize: 16,
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                  SizedBox(height: 5),
                  Text(
                    prato['descricao'],
                    style: AppTextStyles.caption,
                    maxLines: 2,
                    overflow: TextOverflow.ellipsis,
                  ),
                  SizedBox(height: 8),
                  Row(
                    mainAxisAlignment: MainAxisAlignment.spaceBetween,
                    children: [
                      Text(
                        "${prato['preco']} Kz",
                        style: TextStyle(
                          fontSize: 16,
                          fontWeight: FontWeight.bold,
                          color: AppColors.primary,
                        ),
                      ),
                      ElevatedButton(
                        style: ElevatedButton.styleFrom(
                          backgroundColor: AppColors.primary,
                          minimumSize: Size(60, 30),
                          padding: EdgeInsets.symmetric(horizontal: 12),
                        ),
                        onPressed: () {
                          setState(() {
                            carrinho.add(prato);
                          });
                        },
                        child: Text(
                          "Adicionar",
                          style: TextStyle(color: Colors.white, fontSize: 12),
                        ),
                      ),
                    ],
                  ),
                ],
              ),
            ),
          ),
        ],
      ),
    );
  }

  void _finalizarPedido() {
    double total = carrinho.fold(0, (sum, item) => sum + item['preco']);
    
    showModalBottomSheet(
      context: context,
      isScrollControlled: true,
      shape: RoundedRectangleBorder(
        borderRadius: BorderRadius.vertical(top: Radius.circular(20)),
      ),
      builder: (context) {
        return Container(
          padding: EdgeInsets.all(20),
          child: Column(
            mainAxisSize: MainAxisSize.min,
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Text(
                "Finalizar Pedido",
                style: AppTextStyles.heading2,
              ),
              SizedBox(height: 20),
              
              // Resumo do Pedido
              Container(
                padding: EdgeInsets.all(15),
                decoration: BoxDecoration(
                  color: Colors.grey[100],
                  borderRadius: BorderRadius.circular(10),
                ),
                child: Column(
                  children: [
                    for (var item in carrinho)
                      Padding(
                        padding: EdgeInsets.only(bottom: 8),
                        child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            Text(item['nome']),
                            Text("${item['preco']} Kz"),
                          ],
                        ),
                      ),
                    Divider(),
                    Row(
                      mainAxisAlignment: MainAxisAlignment.spaceBetween,
                      children: [
                        Text(
                          "Total:",
                          style: TextStyle(
                            fontSize: 18,
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                        Text(
                          "${total.toStringAsFixed(0)} Kz",
                          style: TextStyle(
                            fontSize: 18,
                            fontWeight: FontWeight.bold,
                            color: AppColors.primary,
                          ),
                        ),
                      ],
                    ),
                  ],
                ),
              ),
              
              SizedBox(height: 20),
              
              // Botão de Pagamento
              ElevatedButton(
                style: ElevatedButton.styleFrom(
                  backgroundColor: AppColors.primary,
                  minimumSize: Size(double.infinity, 55),
                  shape: RoundedRectangleBorder(
                    borderRadius: BorderRadius.circular(10),
                  ),
                ),
                onPressed: () {
                  // Integração Multicaixa Express
                  _pagarComMulticaixa(total);
                },
                child: Row(
                  mainAxisAlignment: MainAxisAlignment.center,
                  children: [
                    Icon(Icons.payment, color: Colors.white),
                    SizedBox(width: 10),
                    Text(
                      "Pagar com Multicaixa Express",
                      style: TextStyle(color: Colors.white, fontSize: 16),
                    ),
                  ],
                ),
              ),
            ],
          ),
        );
      },
    );
  }

  void _pagarComMulticaixa(double total) {
    // Aqui integrará com POST /pagamentos/multicaixa
    Navigator.pop(context); // Fechar modal
    
    showDialog(
      context: context,
      builder: (context) => AlertDialog(
        title: Text("Pagamento Processado"),
        content: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            Icon(Icons.check_circle, color: AppColors.success, size: 60),
            SizedBox(height: 15),
            Text(
              "O seu pedido foi confirmado!\n\nEntrega via Tupuca em ${widget.restaurante['tempoEntrega']}",
              textAlign: TextAlign.center,
            ),
          ],
        ),
        actions: [
          TextButton(
            onPressed: () {
              Navigator.of(context).pop();
              Navigator.of(context).pop(); // Voltar à lista
            },
            child: Text("OK"),
          ),
        ],
      ),
    );
  }
}
