import 'package:flutter/material.dart';
import 'package:google_ml_kit/google_ml_kit.dart';
import 'package:image_picker/image_picker.dart';
import 'services/kyc_service.dart';
import 'utils/constants.dart';

// Nota: Integre 'image_picker' e 'google_ml_kit' aqui para capturar foto e validar BI/selfie no dispositivo.
class VerificacaoDocumentos extends StatefulWidget {
  @override
  State<VerificacaoDocumentos> createState() => _VerificacaoDocumentosState();
}

class _VerificacaoDocumentosState extends State<VerificacaoDocumentos> {
  int _passoAtual = 0; // 0: BI, 1: Selfie, 2: Alvará (apenas empresas)
  bool _biEnviado = false;
  bool _selfieEnviada = false;
  bool _alvaraEnviado = false;
  bool _isEmpresa = false;
  bool _isUploading = false;
  String? _statusKyc;
  String? _ultimoErro;

  final ImagePicker _picker = ImagePicker();
  final KycService _kycService = KycService();

  int get _totalPassos => _isEmpresa ? 3 : 2;

  @override
  void dispose() {
    _kycService.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final bool ultimoPasso = _passoAtual >= _totalPassos - 1;
    final bool passoConcluido = _estaPassoConcluido(_passoAtual);
    final bool todosConcluidos = _todosPassosConcluidos();

    return Scaffold(
      appBar: AppBar(
        title: Text('Verifica a tua Identidade'),
        backgroundColor: AppColors.primary,
      ),
      body: SafeArea(
        child: SingleChildScrollView(
          padding: EdgeInsets.all(AppDimensions.paddingScreen),
          child: Column(
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              _buildBarraProgresso(),
              SizedBox(height: 24),
              _buildToggleEmpresa(),
              SizedBox(height: 20),
              _buildPassoCard(),
              SizedBox(height: 12),
              _buildDicaPasso(),
              if (_passoAtual > 0) ...[
                SizedBox(height: 16),
                TextButton(
                  onPressed: _passoAtual > 0 ? _voltarPasso : null,
                  child: Text('Voltar para o passo anterior', style: TextStyle(color: AppColors.primary, fontWeight: FontWeight.w600)),
                ),
              ],
            ],
          ),
        ),
      ),
      bottomNavigationBar: SafeArea(
        child: Padding(
          padding: EdgeInsets.all(AppDimensions.paddingScreen),
          child: SizedBox(
            width: double.infinity,
            height: AppDimensions.buttonHeight,
            child: ElevatedButton(
              style: ElevatedButton.styleFrom(
                backgroundColor: AppColors.primary,
                shape: RoundedRectangleBorder(
                  borderRadius: BorderRadius.circular(AppDimensions.borderRadiusMedium),
                ),
              ),
                onPressed: _isUploading
                  ? null
                  : ultimoPasso
                    ? (todosConcluidos ? _enviarParaAnalise : null)
                    : (passoConcluido ? _avancarPasso : null),
              child: Text(
                ultimoPasso ? 'ENVIAR PARA ANÁLISE' : 'CONTINUAR',
                style: TextStyle(fontWeight: FontWeight.bold, color: Colors.white),
              ),
            ),
          ),
        ),
      ),
    );
  }

  Widget _buildBarraProgresso() {
    final double progresso = (_passoAtual + 1) / _totalPassos;

    return Column(
      crossAxisAlignment: CrossAxisAlignment.start,
      children: [
        Text(
          'Passo ${_passoAtual + 1} de $_totalPassos',
          style: AppTextStyles.caption.copyWith(color: AppColors.textLight),
        ),
        SizedBox(height: 8),
        ClipRRect(
          borderRadius: BorderRadius.circular(AppDimensions.borderRadiusMedium),
          child: LinearProgressIndicator(
            value: progresso,
            backgroundColor: Colors.grey[300],
            color: AppColors.primary,
            minHeight: 10,
          ),
        ),
      ],
    );
  }

  Widget _buildToggleEmpresa() {
    return Container(
      padding: EdgeInsets.all(15),
      decoration: BoxDecoration(
        color: AppColors.primary.withOpacity(0.05),
        borderRadius: BorderRadius.circular(AppDimensions.borderRadiusMedium),
      ),
      child: Row(
        children: [
          Expanded(
            child: Text(
              'Conta de Empresa (Business)?',
              style: TextStyle(fontWeight: FontWeight.w700, fontSize: 16, color: AppColors.textDark),
            ),
          ),
          Switch(
            value: _isEmpresa,
            activeColor: AppColors.primary,
            onChanged: (value) {
              setState(() {
                _isEmpresa = value;
                if (!_isEmpresa && _passoAtual > 1) {
                  _passoAtual = 1; // Evita mostrar passo 3 quando conta é pessoal
                }
              });
            },
          ),
        ],
      ),
    );
  }

  Widget _buildPassoCard() {
    final bool documentoEnviado = _estaPassoConcluido(_passoAtual);
    final String titulo = _getTituloPasso();
    final String descricao = _getDescricaoPasso();
    final IconData icone = _getIconeDocumento(documentoEnviado);
    final Color corAtiva = documentoEnviado ? AppColors.success : AppColors.primary;

    return GestureDetector(
      onTap: _isUploading ? null : _uploadDocumento,
      child: Container(
        width: double.infinity,
        padding: EdgeInsets.symmetric(horizontal: 16, vertical: 28),
        decoration: BoxDecoration(
          color: Colors.white,
          borderRadius: BorderRadius.circular(AppDimensions.borderRadiusLarge),
          border: Border.all(color: corAtiva, width: 2),
          boxShadow: [
            BoxShadow(color: Colors.black.withOpacity(0.05), blurRadius: 10, offset: Offset(0, 4)),
          ],
        ),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.center,
          children: [
            Text(titulo, style: AppTextStyles.heading2),
            SizedBox(height: 8),
            Text(descricao, style: AppTextStyles.body, textAlign: TextAlign.center),
            SizedBox(height: 24),
            Container(
              width: 110,
              height: 110,
              decoration: BoxDecoration(
                color: corAtiva.withOpacity(0.08),
                shape: BoxShape.circle,
              ),
              child: Icon(icone, size: 54, color: corAtiva),
            ),
            SizedBox(height: 16),
            Text(
              documentoEnviado ? 'Documento carregado com sucesso' : _getCTAForPasso(),
              style: TextStyle(fontWeight: FontWeight.w700, color: corAtiva),
            ),
            SizedBox(height: 6),
            Text(
              'Toque para carregar ou capturar',
              style: AppTextStyles.caption.copyWith(color: AppColors.textLight),
            ),
            if (_isUploading)
              Padding(
                padding: EdgeInsets.only(top: 12),
                child: Row(
                  mainAxisAlignment: MainAxisAlignment.center,
                  children: [
                    SizedBox(width: 18, height: 18, child: CircularProgressIndicator(strokeWidth: 2.2, color: AppColors.primary)),
                    SizedBox(width: 8),
                    Text('A processar...', style: AppTextStyles.caption),
                  ],
                ),
              ),
            if (_ultimoErro != null)
              Padding(
                padding: EdgeInsets.only(top: 8),
                child: Text(_ultimoErro!, style: AppTextStyles.caption.copyWith(color: AppColors.danger)),
              ),
          ],
        ),
      ),
    );
  }

  Widget _buildDicaPasso() {
    return Container(
      width: double.infinity,
      padding: EdgeInsets.all(16),
      decoration: BoxDecoration(
        color: Colors.white,
        borderRadius: BorderRadius.circular(AppDimensions.borderRadiusMedium),
        border: Border.all(color: Colors.grey[300]!),
      ),
      child: Row(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Icon(Icons.lightbulb, color: AppColors.warning),
          SizedBox(width: 10),
          Expanded(
            child: Text(
              _getDicaPasso(),
              style: AppTextStyles.body,
            ),
          ),
        ],
      ),
    );
  }

  String _getTituloPasso() {
    switch (_passoAtual) {
      case 0:
        return 'Passo 1: Foto do BI';
      case 1:
        return 'Passo 2: Selfie com Liveness';
      case 2:
        return 'Passo 3: Alvará / Diário da República';
      default:
        return '';
    }
  }

  String _getDescricaoPasso() {
    switch (_passoAtual) {
      case 0:
        return 'Submete uma foto clara do Bilhete de Identidade. O app deteta bordos e aplica marca d\'água TudoAqui.';
      case 1:
        return 'Tira uma selfie a sorrir ou a piscar para confirmar presença (liveness). Comparamos rosto vs BI.';
      case 2:
        return 'Carrega o PDF do Alvará ou Diário da República para validar a empresa e liberar vendas Business.';
      default:
        return '';
    }
  }

  String _getDicaPasso() {
    switch (_passoAtual) {
      case 0:
        return 'Dica: usa boa luz e fundo neutro para garantir leitura do BI.';
      case 1:
        return 'Dica: olha direto para a câmara e faz um gesto rápido (sorriso/piscadela) para provar que estás presente.';
      case 2:
        return 'Dica: exporta o Alvará em PDF nítido; aceita também Diário da República digital.';
      default:
        return '';
    }
  }

  String _getCTAForPasso() {
    switch (_passoAtual) {
      case 0:
        return 'Tirar Foto do BI (Frente)';
      case 1:
        return 'Tirar Selfie de Segurança';
      case 2:
        return 'Upload do Alvará (PDF)';
      default:
        return '';
    }
  }

  IconData _getIconeDocumento(bool enviado) {
    if (enviado) return Icons.check_circle;
    switch (_passoAtual) {
      case 0:
        return Icons.credit_card;
      case 1:
        return Icons.face_retouching_natural;
      case 2:
        return Icons.picture_as_pdf;
      default:
        return Icons.upload;
    }
  }

  bool _estaPassoConcluido(int passo) {
    switch (passo) {
      case 0:
        return _biEnviado;
      case 1:
        return _selfieEnviada;
      case 2:
        return _alvaraEnviado;
      default:
        return false;
    }
  }

  bool _todosPassosConcluidos() {
    if (_isEmpresa) {
      return _biEnviado && _selfieEnviada && _alvaraEnviado;
    }
    return _biEnviado && _selfieEnviada;
  }

  void _avancarPasso() {
    if (_passoAtual < _totalPassos - 1) {
      setState(() {
        _passoAtual++;
      });
    }
  }

  void _voltarPasso() {
    if (_passoAtual > 0) {
      setState(() {
        _passoAtual--;
      });
    }
  }

  Future<void> _uploadDocumento() async {
    await _processarPassoAtual();
  }

  Future<void> _processarPassoAtual() async {
    if (_isUploading) return;
    setState(() {
      _isUploading = true;
      _ultimoErro = null;
    });

    try {
      if (_passoAtual == 0) {
        await _processarBI();
      } else if (_passoAtual == 1) {
        await _processarSelfie();
      } else {
        await _processarAlvara();
      }
    } catch (e) {
      _ultimoErro = e.toString();
      _mostrarSnack('Erro: ${e.toString()}', isError: true);
    } finally {
      if (mounted) {
        setState(() {
          _isUploading = false;
        });
      }
    }
  }

  Future<void> _processarBI() async {
    final foto = await _capturarImagem(preferFrontal: false);
    if (foto == null) return;

    final textRecognizer = TextRecognizer(script: TextRecognitionScript.latin);
    try {
      final inputImage = InputImage.fromFilePath(foto.path);
      final recognizedText = await textRecognizer.processImage(inputImage);
      if (recognizedText.text.trim().isEmpty) {
        throw Exception('Não conseguimos ler o BI. Tenta outra foto com mais luz.');
      }

      await _kycService.uploadDocumento(ficheiro: foto, endpoint: ApiEndpoints.uploadBI);
      if (!mounted) return;
      setState(() {
        _biEnviado = true;
      });
      _mostrarSnack('BI enviado com sucesso.');
    } finally {
      await textRecognizer.close();
    }
  }

  Future<void> _processarSelfie() async {
    final selfie = await _capturarImagem(preferFrontal: true);
    if (selfie == null) return;

    final options = FaceDetectorOptions(enableClassification: true, minFaceSize: 0.2);
    final detector = FaceDetector(options: options);
    try {
      final inputImage = InputImage.fromFilePath(selfie.path);
      final faces = await detector.processImage(inputImage);
      if (faces.isEmpty) {
        throw Exception('Não encontrámos um rosto. Ajusta a luz e olha para a câmara.');
      }
      final face = faces.first;
      final sorriso = (face.smilingProbability ?? 0) > 0.4;
      final olhosAbertos = (face.leftEyeOpenProbability ?? 0) > 0.3 && (face.rightEyeOpenProbability ?? 0) > 0.3;
      if (!sorriso && !olhosAbertos) {
        throw Exception('Faz um gesto rápido (sorriso/piscadela) para provar liveness.');
      }

      await _kycService.uploadDocumento(ficheiro: selfie, endpoint: ApiEndpoints.uploadSelfie);
      if (!mounted) return;
      setState(() {
        _selfieEnviada = true;
      });
      _mostrarSnack('Selfie validada e enviada.');
    } finally {
      await detector.close();
    }
  }

  Future<void> _processarAlvara() async {
    final doc = await _capturarImagem(preferFrontal: false, fromGallery: true);
    if (doc == null) return;

    await _kycService.uploadDocumento(ficheiro: doc, endpoint: ApiEndpoints.uploadAlvara);
    if (!mounted) return;
    setState(() {
      _alvaraEnviado = true;
    });
    _mostrarSnack('Alvará enviado com sucesso.');
  }

  Future<XFile?> _capturarImagem({required bool preferFrontal, bool fromGallery = false}) async {
    final source = fromGallery ? ImageSource.gallery : ImageSource.camera;
    return _picker.pickImage(source: source, preferredCameraDevice: preferFrontal ? CameraDevice.front : CameraDevice.rear, imageQuality: 85);
  }

  void _mostrarSnack(String mensagem, {bool isError = false}) {
    if (!mounted) return;
    ScaffoldMessenger.of(context).showSnackBar(
      SnackBar(content: Text(mensagem), backgroundColor: isError ? AppColors.danger : AppColors.primary),
    );
  }

  Future<void> _enviarParaAnalise() async {
    await _atualizarStatusKyc();
    if (!mounted) return;

    showDialog(
      context: context,
      builder: (context) => AlertDialog(
        title: Text('Documentos em Análise'),
        content: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            CircularProgressIndicator(color: AppColors.primary),
            SizedBox(height: 16),
            Text(
              _statusKyc != null
                  ? 'Estado atual: ${_statusKyc!}. Receberás notificação quando terminar.'
                  : 'A equipa TudoAqui está a validar os teus documentos. Receberás uma notificação em breve.',
              textAlign: TextAlign.center,
            ),
          ],
        ),
        actions: [
          TextButton(
            onPressed: () => Navigator.of(context).pop(),
            child: Text('OK', style: TextStyle(color: AppColors.primary, fontWeight: FontWeight.bold)),
          ),
        ],
      ),
    );
  }

  Future<void> _atualizarStatusKyc() async {
    try {
      final status = await _kycService.obterStatus();
      if (!mounted) return;
      setState(() {
        _statusKyc = status;
      });
    } catch (e) {
      _mostrarSnack('Não foi possível obter o status: ${e.toString()}', isError: true);
    }
  }
}
