# Script para gerar certificados e chaves de assinatura
# Uso: ./scripts/certificates.ps1

Write-Host "🔐 Gerador de Certificados e Chaves TudoAqui" -ForegroundColor Cyan
Write-Host "===========================================" -ForegroundColor Cyan
Write-Host ""

# ============================================
# 1. CHAVE SSH PARA AWS
# ============================================
Write-Host "📌 1. Gerando SSH Key para AWS..." -ForegroundColor Yellow

$sshDir = "$HOME\.ssh"
$keyPath = "$sshDir\tudoaqui-aws"

if (-not (Test-Path $sshDir)) {
    New-Item -ItemType Directory -Path $sshDir -Force | Out-Null
}

if (Test-Path "$keyPath.pub") {
    Write-Host "⚠️  SSH key já existe em $keyPath" -ForegroundColor Yellow
} else {
    # Usar ssh-keygen se disponível (Windows 10/11 com Git Bash)
    $sshKeygenPath = "C:\Program Files\Git\usr\bin\ssh-keygen.exe"
    
    if (Test-Path $sshKeygenPath) {
        & $sshKeygenPath -t rsa -b 4096 -f $keyPath -N "" -C "tudoaqui@$(hostname)"
        Write-Host "✅ SSH key gerada: $keyPath" -ForegroundColor Green
    } else {
        Write-Host "⚠️  ssh-keygen não encontrado. Instale Git Bash para Windows." -ForegroundColor Red
        Write-Host "   Download: https://git-scm.com/download/win" -ForegroundColor Gray
    }
}

# ============================================
# 2. CHAVE PARA ANDROID
# ============================================
Write-Host ""
Write-Host "📌 2. Gerando keystore para Android..." -ForegroundColor Yellow

$keystoreDir = "$HOME"
$keystorePath = "$keystoreDir\tudoaqui-release.jks"
$alias = "tudoaqui-key"

if (Test-Path $keystorePath) {
    Write-Host "⚠️  Keystore já existe em $keystorePath" -ForegroundColor Yellow
} else {
    $keyPassword = Read-Host "🔑 Digite a senha para o keystore (min 6 caracteres)"
    
    if ($keyPassword.Length -lt 6) {
        Write-Host "❌ Senha muito curta. Mínimo 6 caracteres." -ForegroundColor Red
        exit 1
    }
    
    # Verificar se keytool está disponível (vem com Java)
    $keytoolPath = "C:\Program Files\Java\jdk*\bin\keytool.exe"
    $keytoolFound = Get-Item -Path $keytoolPath -ErrorAction SilentlyContinue | Select-Object -First 1
    
    if ($keytoolFound) {
        Write-Host "📝 Preenchendo certificado para Android..." -ForegroundColor Cyan
        
        $keytoolExe = $keytoolFound.FullName
        
        & $keytoolExe -genkey -v -keystore $keystorePath `
            -keyalg RSA -keysize 2048 -validity 10000 `
            -alias $alias `
            -keypass $keyPassword `
            -storepass $keyPassword `
            -dname "CN=TudoAqui, OU=Engineering, O=TudoAqui, L=Luanda, ST=Luanda, C=AO"
        
        if ($LASTEXITCODE -eq 0) {
            Write-Host "✅ Keystore gerada: $keystorePath" -ForegroundColor Green
            Write-Host "   Alias: $alias" -ForegroundColor Green
            Write-Host "   Password: [guardada com segurança]" -ForegroundColor Green
        } else {
            Write-Host "❌ Erro ao gerar keystore" -ForegroundColor Red
        }
    } else {
        Write-Host "⚠️  keytool não encontrado. Instale Java Development Kit (JDK)." -ForegroundColor Red
        Write-Host "   Download: https://www.oracle.com/java/technologies/downloads/" -ForegroundColor Gray
    }
}

# ============================================
# 3. CERTIFICADO AUTO-ASSINADO PARA DESENVOLVIMENTO
# ============================================
Write-Host ""
Write-Host "📌 3. Gerando certificado auto-assinado para desenvolvimento..." -ForegroundColor Yellow

$certDir = "$PSScriptRoot\..\certificates"
if (-not (Test-Path $certDir)) {
    New-Item -ItemType Directory -Path $certDir -Force | Out-Null
}

$certPath = "$certDir\localhost.crt"
$keyPath = "$certDir\localhost.key"

if ((Test-Path $certPath) -and (Test-Path $keyPath)) {
    Write-Host "⚠️  Certificado já existe" -ForegroundColor Yellow
} else {
    # Usar openssl se disponível
    $opensslPath = "C:\Program Files\Git\usr\bin\openssl.exe"
    
    if (Test-Path $opensslPath) {
        & $opensslPath req -x509 -newkey rsa:4096 -nodes -out $certPath -keyout $keyPath -days 365 `
            -subj "/C=AO/ST=Luanda/L=Luanda/O=TudoAqui/OU=Dev/CN=localhost"
        
        if ($LASTEXITCODE -eq 0) {
            Write-Host "✅ Certificado auto-assinado gerado" -ForegroundColor Green
            Write-Host "   Certificado: $certPath" -ForegroundColor Green
            Write-Host "   Chave privada: $keyPath" -ForegroundColor Green
        }
    } else {
        Write-Host "⚠️  openssl não encontrado. Instale Git Bash." -ForegroundColor Red
    }
}

# ============================================
# 4. GERAR SECRETS PARA .ENV
# ============================================
Write-Host ""
Write-Host "📌 4. Gerando secrets para .env..." -ForegroundColor Yellow

function Generate-RandomString {
    param([int]$Length = 32)
    $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*()'
    $random = 1..$Length | ForEach-Object { $chars | Get-Random }
    return -join $random
}

$jwtSecret = Generate-RandomString 64
$jwtRefreshSecret = Generate-RandomString 64
$webhookSecret = Generate-RandomString 32

Write-Host ""
Write-Host "🔑 Secrets gerados (guardar em .env):" -ForegroundColor Cyan
Write-Host "JWT_SECRET=$jwtSecret" -ForegroundColor Green
Write-Host "JWT_REFRESH_SECRET=$jwtRefreshSecret" -ForegroundColor Green
Write-Host "WEBHOOK_SECRET=$webhookSecret" -ForegroundColor Green

# Salvar em arquivo temporário
$secretsFile = "$certDir\secrets.txt"
@"
# Secrets gerados em $(Get-Date)
JWT_SECRET=$jwtSecret
JWT_REFRESH_SECRET=$jwtRefreshSecret
WEBHOOK_SECRET=$webhookSecret
"@ | Out-File -FilePath $secretsFile -Encoding UTF8

Write-Host ""
Write-Host "✅ Secrets salvos em: $secretsFile" -ForegroundColor Green

# ============================================
# 5. RESUMO
# ============================================
Write-Host ""
Write-Host "📊 Resumo dos Certificados e Chaves" -ForegroundColor Cyan
Write-Host "===================================" -ForegroundColor Cyan
Write-Host ""
Write-Host "SSH (AWS):" -ForegroundColor Yellow
Write-Host "  Caminho: $keyPath" -ForegroundColor Gray
Write-Host "  Usar em Terraform com: key_name = aws_key_pair.main.key_name" -ForegroundColor Gray
Write-Host ""
Write-Host "Android (Play Store):" -ForegroundColor Yellow
Write-Host "  Caminho: $keystorePath" -ForegroundColor Gray
Write-Host "  Alias: $alias" -ForegroundColor Gray
Write-Host "  Usar em: android/app/build.gradle" -ForegroundColor Gray
Write-Host ""
Write-Host "Certificado Self-Signed:" -ForegroundColor Yellow
Write-Host "  Cert: $certPath" -ForegroundColor Gray
Write-Host "  Key: $keyPath" -ForegroundColor Gray
Write-Host "  Uso: Desenvolvimento local apenas" -ForegroundColor Gray
Write-Host ""
Write-Host "Secrets:" -ForegroundColor Yellow
Write-Host "  Arquivo: $secretsFile" -ForegroundColor Gray
Write-Host "  ⚠️  Guardar em segurança - NÃO commitar no Git" -ForegroundColor Red
Write-Host ""

# ============================================
# 6. ADICIONAR AO .gitignore
# ============================================
Write-Host "📝 Adicionando arquivos sensíveis ao .gitignore..." -ForegroundColor Yellow

$gitignorePath = "$PSScriptRoot\..\..\.gitignore"
$filesToIgnore = @(
    "tudoaqui-release.jks",
    "certificates/",
    ".env",
    ".env.local",
    ".env.*.local",
    "secrets.txt"
)

if (Test-Path $gitignorePath) {
    $gitignoreContent = Get-Content $gitignorePath -Raw
    
    foreach ($file in $filesToIgnore) {
        if ($gitignoreContent -notlike "*$file*") {
            Add-Content -Path $gitignorePath -Value "`n$file"
            Write-Host "  ✅ Adicionado: $file" -ForegroundColor Green
        }
    }
} else {
    $filesToIgnore | Out-File -FilePath $gitignorePath -Encoding UTF8
    Write-Host "  ✅ .gitignore criado com arquivos sensíveis" -ForegroundColor Green
}

Write-Host ""
Write-Host "🎉 Certificados e chaves gerados com sucesso!" -ForegroundColor Green
Write-Host ""
Write-Host "📚 Próximos passos:" -ForegroundColor Cyan
Write-Host "  1. Copiar SSH public key para AWS" -ForegroundColor Gray
Write-Host "  2. Configurar android/app/build.gradle com keystore" -ForegroundColor Gray
Write-Host "  3. Preencher .env com todos os secrets" -ForegroundColor Gray
Write-Host "  4. Commitar .env.example (nunca .env real)" -ForegroundColor Gray
Write-Host ""
