#!/bin/bash
set -e

# Script de deployment para TudoAqui
# Uso: ./scripts/deploy.sh [staging|production]

ENVIRONMENT=${1:-staging}
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"

# ============================================
# Cores para output
# ============================================
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# ============================================
# Funções
# ============================================
log_info() {
    echo -e "${BLUE}ℹ️  $1${NC}"
}

log_success() {
    echo -e "${GREEN}✅ $1${NC}"
}

log_warning() {
    echo -e "${YELLOW}⚠️  $1${NC}"
}

log_error() {
    echo -e "${RED}❌ $1${NC}"
}

# ============================================
# Validações
# ============================================
if [ "$ENVIRONMENT" != "staging" ] && [ "$ENVIRONMENT" != "production" ]; then
    log_error "Ambiente inválido. Use: staging ou production"
    exit 1
fi

log_info "🚀 Iniciando deployment para $ENVIRONMENT"

# ============================================
# 1. Validar ambiente
# ============================================
log_info "Validando ambiente..."

if ! command -v node &> /dev/null; then
    log_error "Node.js não instalado"
    exit 1
fi

if ! command -v npm &> /dev/null; then
    log_error "npm não instalado"
    exit 1
fi

log_success "Node.js $(node -v) encontrado"

# ============================================
# 2. Carregar variáveis de ambiente
# ============================================
log_info "Carregando variáveis de ambiente..."

ENV_FILE="$PROJECT_DIR/.env.$ENVIRONMENT"
if [ -f "$ENV_FILE" ]; then
    export $(cat "$ENV_FILE" | grep -v '^#' | xargs)
    log_success "Variáveis carregadas de $ENV_FILE"
else
    log_warning ".env.$ENVIRONMENT não encontrado, usando .env"
    if [ -f "$PROJECT_DIR/.env" ]; then
        export $(cat "$PROJECT_DIR/.env" | grep -v '^#' | xargs)
    else
        log_error ".env não encontrado"
        exit 1
    fi
fi

# ============================================
# 3. Validar credenciais AWS
# ============================================
log_info "Validando credenciais AWS..."

if ! aws sts get-caller-identity &> /dev/null; then
    log_error "Credenciais AWS não configuradas"
    exit 1
fi

log_success "Credenciais AWS válidas"

# ============================================
# 4. Build
# ============================================
log_info "Executando build..."

cd "$PROJECT_DIR"
npm ci --production
npm run build

if [ $? -eq 0 ]; then
    log_success "Build concluído com sucesso"
else
    log_error "Build falhou"
    exit 1
fi

# ============================================
# 5. Tests (opcional para production)
# ============================================
if [ "$ENVIRONMENT" = "staging" ]; then
    log_info "Executando testes..."
    npm test -- --coverage
    
    if [ $? -eq 0 ]; then
        log_success "Todos os testes passaram"
    else
        log_warning "Alguns testes falharam, continuando..."
    fi
fi

# ============================================
# 6. Docker build
# ============================================
log_info "Building Docker image..."

IMAGE_NAME="tudoaqui-api"
IMAGE_TAG="$ENVIRONMENT-$(date +%Y%m%d-%H%M%S)"
FULL_IMAGE="$IMAGE_NAME:$IMAGE_TAG"

docker build -t "$FULL_IMAGE" -f Dockerfile .

if [ $? -eq 0 ]; then
    log_success "Docker image criada: $FULL_IMAGE"
else
    log_error "Docker build falhou"
    exit 1
fi

# ============================================
# 7. Push para ECR
# ============================================
log_info "Pushing image para AWS ECR..."

AWS_REGION=$(aws configure get region)
AWS_ACCOUNT_ID=$(aws sts get-caller-identity --query Account --output text)
ECR_REGISTRY="$AWS_ACCOUNT_ID.dkr.ecr.$AWS_REGION.amazonaws.com"
ECR_IMAGE="$ECR_REGISTRY/tudoaqui:$IMAGE_TAG"

# Login no ECR
aws ecr get-login-password --region "$AWS_REGION" | \
    docker login --username AWS --password-stdin "$ECR_REGISTRY"

# Tag e push
docker tag "$FULL_IMAGE" "$ECR_IMAGE"
docker push "$ECR_IMAGE"

log_success "Image enviada para ECR: $ECR_IMAGE"

# ============================================
# 8. Deploy em EC2
# ============================================
log_info "Deployando em EC2..."

if [ "$ENVIRONMENT" = "staging" ]; then
    EC2_HOST=${EC2_STAGING_HOST}
else
    EC2_HOST=${EC2_PROD_HOST}
fi

if [ -z "$EC2_HOST" ]; then
    log_error "EC2_HOST não configurado para $ENVIRONMENT"
    exit 1
fi

# SSH deploy
ssh -i ~/.ssh/tudoaqui-aws ubuntu@"$EC2_HOST" << EOF
    set -e
    cd /var/www/tudoaqui
    
    log_info "Parando container anterior..."
    docker stop tudoaqui-api || true
    docker rm tudoaqui-api || true
    
    log_info "Puxando nova image..."
    docker pull "$ECR_IMAGE"
    
    log_info "Iniciando novo container..."
    docker run -d \
        --name tudoaqui-api \
        -p 3000:3000 \
        --env-file .env \
        --restart unless-stopped \
        -v /var/log/tudoaqui:/app/logs \
        "$ECR_IMAGE"
    
    log_info "Esperando aplicação iniciar..."
    sleep 10
    
    # Health check
    for i in {1..10}; do
        if curl -f http://localhost:3000/health > /dev/null 2>&1; then
            log_success "✅ Aplicação respondendo"
            exit 0
        fi
        log_info "Tentativa \$i/10..."
        sleep 5
    done
    
    log_error "❌ Health check falhou"
    exit 1
EOF

if [ $? -eq 0 ]; then
    log_success "Deploy concluído com sucesso"
else
    log_error "Deploy falhou"
    exit 1
fi

# ============================================
# 9. Verificação final
# ============================================
log_info "Realizando verificação final..."

if [ "$ENVIRONMENT" = "staging" ]; then
    API_URL="https://staging-api.tudoaqui.ao"
else
    API_URL="https://api.tudoaqui.ao"
fi

for i in {1..5}; do
    if curl -f "$API_URL/health" > /dev/null 2>&1; then
        log_success "✅ API respondendo: $API_URL"
        break
    fi
    log_warning "Tentativa $i/5..."
    sleep 5
done

# ============================================
# 10. Notificação
# ============================================
log_success "🎉 Deployment para $ENVIRONMENT concluído!"
log_info "API: $API_URL"
log_info "Swagger: $API_URL/api-docs"
log_info "Health: $API_URL/health"
