# PowerShell script para setup Git e GitHub no Windows

Write-Host "🚀 TudoAqui GitHub Setup (Windows)" -ForegroundColor Cyan
Write-Host "===================================" -ForegroundColor Cyan
Write-Host ""

# ============================================
# 1. Verificar Git
# ============================================
Write-Host "📌 Verificando Git..." -ForegroundColor Yellow

if (!(Get-Command git -ErrorAction SilentlyContinue)) {
    Write-Host "❌ Git não encontrado. Instale: https://git-scm.com/download/win" -ForegroundColor Red
    exit 1
}

$gitVersion = git --version
Write-Host "✅ $gitVersion" -ForegroundColor Green

# ============================================
# 2. Configurar Git
# ============================================
Write-Host ""
Write-Host "📌 Configurando Git local..." -ForegroundColor Yellow

$gitName = Read-Host "👤 Seu nome (para commits)"
$gitEmail = Read-Host "📧 Seu email (para commits)"

git config user.name "$gitName"
git config user.email "$gitEmail"

Write-Host "✅ Git configurado" -ForegroundColor Green

# ============================================
# 3. Inicializar repositório
# ============================================
Write-Host ""
Write-Host "📌 Inicializando repositório..." -ForegroundColor Yellow

if (!(Test-Path ".\.git")) {
    git init
    Write-Host "✅ Repositório inicializado" -ForegroundColor Green
} else {
    Write-Host "⚠️  Repositório já existe" -ForegroundColor Yellow
}

# ============================================
# 4. Criar branches
# ============================================
Write-Host ""
Write-Host "📌 Configurando branches..." -ForegroundColor Yellow

# Main
try {
    git show-ref --quiet refs/heads/main
} catch {
    git checkout -b main 2>$null
}

# Develop
try {
    git show-ref --quiet refs/heads/develop
} catch {
    git checkout -b develop 2>$null
}

git checkout main 2>$null
Write-Host "✅ Branches 'main' e 'develop' criadas" -ForegroundColor Green

# ============================================
# 5. Primeiro commit
# ============================================
Write-Host ""
Write-Host "📌 Preparando primeiro commit..." -ForegroundColor Yellow

git add .
$status = git status --porcelain
if ($status) {
    git commit -m "🚀 Lançamento inicial TudoAqui - Backend pronto"
    Write-Host "✅ Primeiro commit realizado" -ForegroundColor Green
} else {
    Write-Host "⚠️  Sem mudanças para commitar" -ForegroundColor Yellow
}

# ============================================
# 6. GitHub Setup
# ============================================
Write-Host ""
Write-Host "📌 GitHub Setup" -ForegroundColor Yellow

$createRepo = Read-Host "🎯 Criar repositório no GitHub? (s/n)"

if ($createRepo -eq "s") {
    $repoName = Read-Host "📝 Nome do repositório"
    $githubUser = Read-Host "👥 Seu username GitHub"
    
    Write-Host "🔓 Abra seu navegador para criar o repositório:" -ForegroundColor Cyan
    Write-Host "https://github.com/new" -ForegroundColor Blue
    Write-Host ""
    Write-Host "Após criar, configure o remote:" -ForegroundColor Gray
    Write-Host "git remote add origin https://github.com/$githubUser/$repoName.git" -ForegroundColor Gray
    Write-Host "git branch -M main" -ForegroundColor Gray
    Write-Host "git push -u origin main" -ForegroundColor Gray
}

# ============================================
# 7. GitHub CLI
# ============================================
Write-Host ""
Write-Host "📌 GitHub CLI" -ForegroundColor Yellow

if (Get-Command gh -ErrorAction SilentlyContinue) {
    Write-Host "✅ GitHub CLI encontrado" -ForegroundColor Green
    
    $setupSecrets = Read-Host "🔐 Configurar GitHub Secrets? (s/n)"
    
    if ($setupSecrets -eq "s") {
        Write-Host "Leia: ./SECURITY_GUIDE.md para instruções de secrets" -ForegroundColor Yellow
    }
} else {
    Write-Host "⚠️  GitHub CLI não encontrado" -ForegroundColor Yellow
    Write-Host "Instale com: choco install gh" -ForegroundColor Gray
}

# ============================================
# 8. Resumo
# ============================================
Write-Host ""
Write-Host "✅ Setup Concluído!" -ForegroundColor Green
Write-Host ""
Write-Host "📊 Resumo:" -ForegroundColor Cyan
Write-Host "  Git: $gitName <$gitEmail>" -ForegroundColor Gray
Write-Host "  Branches: main, develop" -ForegroundColor Gray
Write-Host "  Status: $(git status -s | Measure-Object -Line | Select-Object -ExpandProperty Lines) mudanças" -ForegroundColor Gray
Write-Host ""
Write-Host "🚀 Próximos passos:" -ForegroundColor Cyan
Write-Host "  1. Ir para GitHub e criar repositório" -ForegroundColor Gray
Write-Host "  2. Adicionar remote: git remote add origin https://..." -ForegroundColor Gray
Write-Host "  3. Push: git push -u origin main" -ForegroundColor Gray
Write-Host "  4. Configurar AWS Secrets em GitHub" -ForegroundColor Gray
Write-Host ""
Write-Host "📖 Documentação: GITHUB_SETUP.md" -ForegroundColor Yellow
