#!/bin/bash
# Script para setup Git e GitHub automaticamente

set -e

# Cores
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Funções
log_info() { echo -e "${BLUE}ℹ️  $1${NC}"; }
log_success() { echo -e "${GREEN}✅ $1${NC}"; }
log_warning() { echo -e "${YELLOW}⚠️  $1${NC}"; }
log_error() { echo -e "${RED}❌ $1${NC}"; }

echo -e "${BLUE}"
echo "🚀 TudoAqui GitHub Setup"
echo "======================="
echo -e "${NC}"

# ============================================
# 1. Validar Git
# ============================================
log_info "Verificando Git..."

if ! command -v git &> /dev/null; then
    log_error "Git não encontrado. Instale Git primeiro."
    exit 1
fi

log_success "Git encontrado: $(git --version)"

# ============================================
# 2. Configurar Git Local
# ============================================
log_info "Configurando Git local..."

read -p "👤 Seu nome (para commits): " GIT_NAME
read -p "📧 Seu email (para commits): " GIT_EMAIL

git config user.name "$GIT_NAME"
git config user.email "$GIT_EMAIL"

log_success "Git configurado com:"
echo "  Nome: $GIT_NAME"
echo "  Email: $GIT_EMAIL"

# ============================================
# 3. Inicializar repositório
# ============================================
log_info "Inicializando repositório Git..."

if [ -d ".git" ]; then
    log_warning "Repositório Git já existe"
else
    git init
    log_success "Repositório Git inicializado"
fi

# ============================================
# 4. Criar branches
# ============================================
log_info "Criando branches..."

# Main branch
if ! git rev-parse --quiet --verify main &> /dev/null; then
    git checkout -b main 2>/dev/null || true
    log_success "Branch 'main' criada"
fi

# Develop branch
if ! git rev-parse --quiet --verify develop &> /dev/null; then
    git checkout -b develop 2>/dev/null || true
    log_success "Branch 'develop' criada"
fi

git checkout main 2>/dev/null || true

# ============================================
# 5. Setup GitHub (CLI)
# ============================================
log_info "Verificando GitHub CLI..."

if command -v gh &> /dev/null; then
    log_success "GitHub CLI encontrado: $(gh --version)"
    
    # Autenticar se necessário
    if ! gh auth status &> /dev/null; then
        log_warning "GitHub não autenticado"
        read -p "🔓 Quer fazer login no GitHub agora? (y/n): " GITHUB_LOGIN
        
        if [ "$GITHUB_LOGIN" = "y" ]; then
            gh auth login
        fi
    else
        log_success "GitHub autenticado"
    fi
    
    # Criar repositório
    read -p "🎯 Quer criar repositório no GitHub? (y/n): " CREATE_REPO
    
    if [ "$CREATE_REPO" = "y" ]; then
        read -p "📝 Nome do repositório (ex: tudoaqui): " REPO_NAME
        read -p "👥 Seu username GitHub: " GITHUB_USER
        
        REPO_URL="https://github.com/$GITHUB_USER/$REPO_NAME"
        
        # Criar repositório
        log_info "Criando repositório $REPO_NAME..."
        
        if gh repo create "$REPO_NAME" --public \
            --description "O super app de Angola - Encontre tudo num só lugar" \
            --homepage "https://tudoaqui.ao" \
            --source=. \
            --remote=origin \
            --push 2>/dev/null; then
            
            log_success "Repositório criado: $REPO_URL"
        else
            log_warning "Repositório pode já existir"
            git remote add origin "$REPO_URL" 2>/dev/null || true
        fi
    fi
else
    log_warning "GitHub CLI não encontrado"
    log_info "Instale com: https://cli.github.com"
fi

# ============================================
# 6. Fazer primeiro commit
# ============================================
log_info "Preparando primeiro commit..."

# Adicionar todos os ficheiros
git add .

# Verificar se há mudanças
if git diff --cached --quiet; then
    log_warning "Sem mudanças para commitar"
else
    log_info "Commitando mudanças..."
    git commit -m "🚀 Lançamento inicial TudoAqui - Backend pronto"
    log_success "Primeiro commit realizado"
fi

# ============================================
# 7. Configurar GitHub Secrets
# ============================================
if command -v gh &> /dev/null && gh auth status &> /dev/null; then
    log_info "Configurar GitHub Secrets? (necessário para CI/CD)"
    
    read -p "🔐 Quer configurar secrets agora? (y/n): " SETUP_SECRETS
    
    if [ "$SETUP_SECRETS" = "y" ]; then
        log_info "Adicionando secrets..."
        
        # Ler from arquivo de secrets
        if [ -f "certificates/secrets.txt" ]; then
            read -p "🔑 Path para AWS credentials: " AWS_CREDS_PATH
            
            if [ -f "$AWS_CREDS_PATH" ]; then
                AWS_KEY=$(grep "aws_access_key" "$AWS_CREDS_PATH" | cut -d= -f2)
                AWS_SECRET=$(grep "aws_secret_key" "$AWS_CREDS_PATH" | cut -d= -f2)
                
                # Adicionar secrets
                gh secret set AWS_ACCESS_KEY_ID --body "$AWS_KEY" 2>/dev/null && \
                    log_success "AWS_ACCESS_KEY_ID adicionado"
                
                gh secret set AWS_SECRET_ACCESS_KEY --body "$AWS_SECRET" 2>/dev/null && \
                    log_success "AWS_SECRET_ACCESS_KEY adicionado"
            fi
        fi
        
        log_info "Adicione mais secrets manualmente:"
        log_info "  gh secret set KEY --body 'value'"
    fi
fi

# ============================================
# 8. Resumo Final
# ============================================
echo ""
log_success "✅ Setup GitHub concluído!"
echo ""
echo -e "${BLUE}📊 Resumo:${NC}"
echo "  Git config: $GIT_NAME <$GIT_EMAIL>"
echo "  Branches: main, develop"
echo "  Remote: $(git remote get-url origin 2>/dev/null || echo 'Não configurado')"
echo ""
echo -e "${BLUE}🚀 Próximos passos:${NC}"
echo "  1. Configurar AWS credentials em GitHub Secrets"
echo "  2. Preencher terraform.tfvars com seus valores"
echo "  3. Push das mudanças: git push origin main develop"
echo "  4. Verificar CI/CD workflows em Actions"
echo ""
echo -e "${YELLOW}📝 Documentação:${NC}"
echo "  - QUICK_DEPLOY_GUIDE.md"
echo "  - GITHUB_SETUP.md"
echo "  - SECURITY_GUIDE.md"
echo ""
