const http = require('http');
const fs = require('fs');
const path = require('path');

const BASE_URL = 'http://localhost:3000/api/v1';

function makeRequest(method, endpoint, data = null, token = null) {
  return new Promise((resolve, reject) => {
    const url = new URL(BASE_URL + endpoint);
    const options = {
      hostname: url.hostname,
      port: url.port,
      path: url.pathname + url.search,
      method: method,
      headers: {
        'Content-Type': 'application/json',
      }
    };
    
    if (token) {
      options.headers['Authorization'] = `Bearer ${token}`;
    }

    const req = http.request(options, (res) => {
      let responseData = '';
      res.on('data', (chunk) => responseData += chunk);
      res.on('end', () => {
        try {
          resolve({
            status: res.statusCode,
            data: JSON.parse(responseData)
          });
        } catch (e) {
          resolve({
            status: res.statusCode,
            data: responseData
          });
        }
      });
    });

    req.on('error', reject);
    
    if (data) {
      req.write(JSON.stringify(data));
    }
    req.end();
  });
}

async function runTests() {
  console.log('🧪 TESTES KYC BACKEND\n');
  
  try {
    // 1. Registar
    console.log('=' .repeat(60));
    console.log('1️⃣ REGISTAR UTILIZADOR');
    console.log('='.repeat(60));
    
    const registerData = {
      email: 'teste@tudoaqui.ao',
      senha: 'Senha123!',
      nome: 'Teste Utilizador',
      telefone: '+244912345678',
      tipo: 'vendedor'
    };
    
    const registerRes = await makeRequest('POST', '/auth/register', registerData);
    console.log(`Status: ${registerRes.status}`);
    console.log(`Response:`, registerRes.data);
    
    if (registerRes.status !== 201) {
      console.log('❌ Falha ao registar');
      return;
    }
    
    const userId = registerRes.data.id || registerRes.data.userId;
    const token = registerRes.data.access_token;
    
    console.log(`\n✅ Utilizador criado!`);
    console.log(`User ID: ${userId}`);
    console.log(`Token: ${token ? token.substring(0, 50) + '...' : 'NENHUM'}`);
    
    // 2. Status KYC
    console.log('\n' + '='.repeat(60));
    console.log('2️⃣ VERIFICAR STATUS KYC INICIAL');
    console.log('='.repeat(60));
    
    const statusRes = await makeRequest('GET', `/kyc/status/${userId}`, null, token);
    console.log(`Status: ${statusRes.status}`);
    console.log(`Response:`, statusRes.data);
    
    if (statusRes.status === 200) {
      console.log(`\n📊 STATUS KYC INICIAL:`);
      console.log(`  kycStatus: ${statusRes.data.kycStatus}`);
      console.log(`  seloVerificado: ${statusRes.data.seloVerificado}`);
      console.log(`  biNumero: ${statusRes.data.biNumero || 'N/A'}`);
    }
    
    console.log('\n✅ TESTES CONCLUÍDOS COM SUCESSO!');
    
  } catch (error) {
    console.error('❌ Erro:', error.message);
  }
}

runTests();
