# ============================================
# VALIDACAO TERRAFORM - TudoAqui
# ============================================

Write-Host "`n======================================" -ForegroundColor Cyan
Write-Host "  VALIDACAO TERRAFORM" -ForegroundColor Cyan  
Write-Host "======================================`n" -ForegroundColor Cyan

$ErrorCount = 0
$WarningCount = 0

# 1. Verificar ficheiros
Write-Host "[1/4] Verificando ficheiros..." -ForegroundColor Yellow

$files = @("main.tf", "variables.tf", "outputs.tf", "terraform.tfvars")
foreach ($f in $files) {
    if (Test-Path "infrastructure\$f") {
        Write-Host "  [OK] $f" -ForegroundColor Green
    } else {
        Write-Host "  [ERRO] $f nao encontrado" -ForegroundColor Red
        $ErrorCount++
    }
}

# 2. Validar terraform.tfvars
Write-Host "`n[2/4] Validando terraform.tfvars..." -ForegroundColor Yellow

if (Test-Path "infrastructure\terraform.tfvars") {
    $tfvars = Get-Content "infrastructure\terraform.tfvars" -Raw
    
    # Verificar campos criticos
    if ($tfvars -match "db_password") {
        Write-Host "  [OK] db_password definido" -ForegroundColor Green
        
        # Verificar se nao e valor de exemplo
        if ($tfvars -match 'db_password\s*=\s*"([^"]+)"') {
            $pwd = $matches[1]
            if ($pwd.Length -lt 16) {
                Write-Host "  [!] db_password tem menos de 16 caracteres" -ForegroundColor Yellow
                $WarningCount++
            }
            if ($pwd -match "ChangeMe|example|test|password123") {
                Write-Host "  [!] db_password parece ser valor de exemplo!" -ForegroundColor Yellow
                $WarningCount++
            }
        }
    } else {
        Write-Host "  [ERRO] db_password nao encontrado" -ForegroundColor Red
        $ErrorCount++
    }
    
    if ($tfvars -match "aws_region") {
        Write-Host "  [OK] aws_region definido" -ForegroundColor Green
    } else {
        Write-Host "  [ERRO] aws_region nao definido" -ForegroundColor Red
        $ErrorCount++
    }
    
    if ($tfvars -match "app_name.*tudoaqui") {
        Write-Host "  [OK] app_name = tudoaqui" -ForegroundColor Green
    }
    
} else {
    Write-Host "  [ERRO] terraform.tfvars nao existe!" -ForegroundColor Red
    Write-Host "  Copie: cp terraform.tfvars.example terraform.tfvars" -ForegroundColor Cyan
    $ErrorCount++
}

# 3. Validar sintaxe HCL
Write-Host "`n[3/4] Validando sintaxe HCL..." -ForegroundColor Yellow

if (Test-Path "infrastructure\main.tf") {
    $content = Get-Content "infrastructure\main.tf" -Raw
    
    $openBraces = ($content -split '\{').Count - 1
    $closeBraces = ($content -split '\}').Count - 1
    
    if ($openBraces -eq $closeBraces) {
        Write-Host "  [OK] Chaves balanceadas ($openBraces pares)" -ForegroundColor Green
    } else {
        Write-Host "  [ERRO] Chaves desbalanceadas!" -ForegroundColor Red
        $ErrorCount++
    }
}

# 4. Verificar recursos AWS
Write-Host "`n[4/4] Verificando recursos AWS..." -ForegroundColor Yellow

if (Test-Path "infrastructure\main.tf") {
    $mainTf = Get-Content "infrastructure\main.tf" -Raw
    
    $resources = @{
        'aws_vpc' = 'VPC'
        'aws_instance' = 'EC2 Instance'
        'aws_db_instance' = 'RDS Database'
        'aws_elasticache' = 'Redis Cache'
        'aws_s3_bucket' = 'S3 Bucket'
    }
    
    foreach ($r in $resources.Keys) {
        if ($mainTf -match $r) {
            Write-Host "  [OK] $($resources[$r])" -ForegroundColor Green
        } else {
            Write-Host "  [!] $($resources[$r]) nao encontrado" -ForegroundColor Yellow
            $WarningCount++
        }
    }
}

# Resumo final
Write-Host "`n======================================" -ForegroundColor Cyan
Write-Host "  RESUMO" -ForegroundColor Cyan
Write-Host "======================================`n" -ForegroundColor Cyan

if ($ErrorCount -eq 0 -and $WarningCount -eq 0) {
    Write-Host "SUCCESS: Validacao completa!" -ForegroundColor Green
    Write-Host "`nProximos comandos:" -ForegroundColor Cyan
    Write-Host "  cd infrastructure" -ForegroundColor White
    Write-Host "  terraform init" -ForegroundColor White
    Write-Host "  terraform plan -out=tfplan" -ForegroundColor White
    Write-Host "  terraform apply tfplan`n" -ForegroundColor White
} elseif ($ErrorCount -eq 0) {
    Write-Host "WARNING: $WarningCount avisos" -ForegroundColor Yellow
    Write-Host "Pode prosseguir mas revise os avisos.`n" -ForegroundColor Yellow
} else {
    Write-Host "ERROR: $ErrorCount erros, $WarningCount avisos" -ForegroundColor Red
    Write-Host "Corrija os erros antes de continuar.`n" -ForegroundColor Red
}

exit $ErrorCount
